//  ____      __     
// |  __|____|  |__  
// |  _|  _  |  _  | Florian & Alain's Blender
// |_| |___._|_____| 
//
// Florian Bonnet & Alain Brobecker, 2009-2011

#define ALAIN

#include <stdio.h>  // file functions, etc... 
#include <stdlib.h> // exit(), malloc() 
#include <string.h> // string handling 
#include <math.h>   // mathematics
#include <time.h>

#define FALSE 0
#define TRUE 1
#define pi 3.1415926535897932384626433832795

//**** GLOBAL VARIABLES *****************************************************
unsigned char ColorPalette[256][3];
FILE *filep,*fopen(const char *,const char *);

struct IMAGE {
  unsigned short *Adress;
  unsigned int Length;
};
typedef struct IMAGE IMAGE;
IMAGE Img0,Img1,Img2;

// IMPORTANT: The number of bytes in one line MUST always be a multiple of 4,
// so we must sometimes append 0s to align.
unsigned char *BmpAdress;
unsigned int BmpLength;
unsigned char BmpHeader[54]={
  'B','M',  // 0: always "BM"
  0,0,0,0,  // 2: size of file
  0,0,0,0,  // 6:always set to 0, reserved (?)
  54,0,0,0, //10: offset from start of file to bitmap data
  40,0,0,0, //14: size of Infos (54-14)
  0,0,0,0,  //18: width of image
  0,0,0,0,  //22: height of image
  1,0,      //26: nb of planes (?), always 1
  24,0,     //28: nb of bitplanes in {1;4;8;24}
  0,0,0,0,  //30: compression mode (0=none)
  0,0,0,0,  //34: uncompressed size of bitmap data (0 if no compression is ok)
  0,0,0,0,  //38: horizontal pixels per meter
  0,0,0,0,  //42: vertical pixels per meter
  0,0,0,0,  //46: number of colors used in the bitmap (may be 0)
  0,0,0,0   //50: number of colors important (0=>all )
};

//**** ROUTINES *************************************************************


//**** rnd ****
// Returns an unsigned number in [rndmin;rndmax]. 
// Generator from D.E.Knuth's Art of Computer Programming, vol.2, p185. 
// Period of pseudorandom sequence is 2147483646. 
// Using modulo is mathematically incorrect but who cares? 
#define rndM 2147483647 // 2^31-1, a Mersenne prime 
#define rndA 48271      // this does well in the spectral test 
#define rndQ 44488      // rndM/rndA 
#define rndR 3399       // rndM%rndA, it's important that rndR<rndQ 
long rndX;              // should be initialised to 0<rndX<rndM 

unsigned int rnd(rndmin,rndmax) unsigned int rndmin,rndmax; {
  rndX=rndA*(rndX%rndQ)-rndR*(long)(rndX/rndQ); // X=AX mod M 
  if(rndX<0) {rndX+=rndM; }
  return( rndmin+((rndX)%(rndmax-rndmin+1)));
}          

//**** division with rounding ****
// Please note that 255:2 = 127.5 must be rounded to 127 in case we make 127*2
int divide(dividend,divisor) int dividend,divisor; {
  int quotient;
  if(divisor==0) { printf("Division by 0! "); return(0); }
  if(divisor<0) { divisor=-divisor; dividend=-dividend; }
  quotient=dividend/divisor;
  if((dividend%divisor)>(divisor/2)) { quotient++; }
  return(quotient);
}

//**** power2 ****
int power2(n) int n; { if(n>0) { return(2*power2(n-1)); } else { return(1); } }

void ErrorInFile(name,text) char *name,*text; {printf("Error: %s %s\n",name,text); exit(1);}

void CheckUsage(c,n,s1,s2) int c,n; char *s1; char *s2; {
  if(c<n) { printf("Usage: fab %s %s\n",s1,s2); exit(1); }
  printf("%s: ",s1);
}

//**** Word/LongWord Handling for BMP files ****
void WriteLongWord(value,pos,adress) unsigned int value,pos; unsigned char *adress; {
  adress[pos]=value & 255; value=value>>8;
  adress[pos+1]=value & 255; value=value>>8;
  adress[pos+2]=value & 255; value=value>>8;
  adress[pos+3]=value & 255; value=value>>8;
}

void WriteWord(value,pos,adress) unsigned int value,pos; unsigned char *adress; {
  adress[pos]=value & 255; value=value>>8;
  adress[pos+1]=value & 255; value=value>>8;
}

int ReadLongWord(pos,adress) unsigned int pos; unsigned char *adress; {
  return(((adress[pos+3]*256+adress[pos+2])*256+adress[pos+1])*256+adress[pos]);
}

int ReadWord(pos,adress) unsigned int pos; unsigned char *adress; {
  return(adress[pos+1]*256+adress[pos]);
}

//**** Allocate Memory for w*h grey image ****
IMAGE AllocMemForGreyImage(width,height) int width,height; {
  IMAGE Img;
  if((width<0) || (width>65535)) { ErrorInFile("image","too large"); }
  if((height<0) || (height>65535)) { ErrorInFile("image","too high"); }
  Img.Length=(2+width*height)*sizeof(unsigned short); /* size in bytes */
  Img.Adress=(unsigned short *) malloc(Img.Length);
  Img.Adress[0]=width;
  Img.Adress[1]=height;
  return(Img);
}

//**** Allocate Memory and Load Grey Image File ****
IMAGE AllocMemAndLoadGreyImage(name) char *name; {
  IMAGE Img;
  filep=fopen(name,"rb"); // rb = read binary
  if(filep==0) { ErrorInFile(name,"not found"); }
  fseek(filep,0,2); // set the file pointer at the end of file
  Img.Length=(int) ftell(filep); // returns the length of the file in bytes
  fseek(filep,0,0); // set the file pointer at the start of file
  Img.Adress=(unsigned short *) malloc(Img.Length); // allocate memory
  fread(Img.Adress,1,Img.Length,filep); // load file, 1=size of a byte
  fclose(filep);
  return(Img);
}

//**** Save Image ****
void SaveImage(name,Img) char *name; IMAGE Img; {
  unsigned int FileLength;
  filep=fopen(name,"wb"); // write binary
  FileLength=fwrite(Img.Adress,1,Img.Length,filep);
  fclose(filep);
  if(FileLength!=Img.Length) { ErrorInFile(name,"not saved"); }
}

//**** searchmincolinImg0 ****
unsigned short searchmincolorinImg0() {
  int x,y;
  unsigned short width0,height0,mincol,currentcol;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  mincol=Img0.Adress[2];
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      currentcol=Img0.Adress[2+y*width0+x];
      if(currentcol<mincol) { mincol=currentcol; }
    }
  }
  return (mincol);  
}

//**** searchmaxcolinImg0 ****
unsigned short searchmaxcolorinImg0() {
  int x,y;
  unsigned short width0,height0,maxcol,currentcol;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  maxcol=Img0.Adress[2];
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      currentcol=Img0.Adress[2+y*width0+x];
      if(currentcol>maxcol) { maxcol=currentcol; }
    }
  }
  return (maxcol);  
}

//******************************************************************************
//************************** FAB FUNCTIONS *************************************
//******************************************************************************

//**** bmp2rgb ****
void bmp2rgb(name) char *name; {
  unsigned int width,realbmpwidth,height,nb_bpp;
  unsigned short pixel;
  int pos,x,y;
  // load bmp file
  filep=fopen(name,"rb"); // rb = read binary
  if(filep==0) { ErrorInFile(name,"not found"); }
  fseek(filep,0,2); // set the file pointer at the end of file
  BmpLength=(int) ftell(filep); // returns the length of the file in bytes
  fseek(filep,0,0); // set the file pointer at the start of file
  BmpAdress=(unsigned char *) malloc(BmpLength); // allocate memory
  fread(BmpAdress,1,BmpLength,filep); // load file, 1=size of a byte
  fclose(filep);
  // get width & height and check image validity
  if(BmpLength<=54) { ErrorInFile(name,"must be more than 54 bytes"); }
  if((BmpAdress[0]!='B') || (BmpAdress[1]!='M')) { ErrorInFile(name,"isn't a .bmp file"); }
  if(BmpLength!=ReadLongWord(2,BmpAdress)) { ErrorInFile(name,"file corrupted"); }
  if(ReadWord(30,BmpAdress)!=0) { ErrorInFile(name,"is a compressed .bmp file (not yet supported)"); }
  pos=ReadLongWord(10,BmpAdress); //start of data
  if(pos<54) { ErrorInFile(name,"file corrupted"); }
  width=ReadLongWord(18,BmpAdress);
  height=ReadLongWord(22,BmpAdress);
  if((width<0) || (width>65535)) { ErrorInFile(name,"too large"); }
  if((height<0) || (height>65535)) { ErrorInFile(name,"too high"); }
  printf("%d x %d",width,height);
  Img0=AllocMemForGreyImage(width,height); // allocate memory for rgb channels 
  Img1=AllocMemForGreyImage(width,height);
  Img2=AllocMemForGreyImage(width,height);
  // convert image
  nb_bpp=ReadWord(28,BmpAdress);
  if(nb_bpp==24) {
    // convert 24 bpp to 3 channels red.g green.g blue.g in Img0 Img1 Img2
    realbmpwidth=3*width;
    while((realbmpwidth & 3)!=0) { realbmpwidth++; }
    if(pos+3*width*height>BmpLength){ ErrorInFile(name,"file corrupted"); }
    for(y=0;y<height;y++) {
      for(x=0;x<width;x++) {
        Img0.Adress[2+y*width+x]=(unsigned short) BmpAdress[pos+y*realbmpwidth+x*3+2]; // red
        Img1.Adress[2+y*width+x]=(unsigned short) BmpAdress[pos+y*realbmpwidth+x*3+1]; // green
        Img2.Adress[2+y*width+x]=(unsigned short) BmpAdress[pos+y*realbmpwidth+x*3+0]; // blue
      }
    }
  } else {
    // convert ? bpp+palette to 3 channels red.g green.g blue.g in Img0 Img1 Img2
    realbmpwidth=width;
    while(((realbmpwidth*nb_bpp) & 31)!=0) { realbmpwidth++; }
    for(y=0;y<height;y++) {
      for(x=0;x<width;x++) {
        switch(nb_bpp) {
          case 8: pixel=BmpAdress[pos+y*realbmpwidth+x]; break;
          case 4: pixel=((BmpAdress[pos+y*realbmpwidth/2+x/2])>>(4*(1-x%2))) & 15; break;
          case 1: pixel=((BmpAdress[pos+y*realbmpwidth/8+x/8])>>(7-x%8)) & 1; break;
          default: ErrorInFile(name,"has an invalid number of bits per pixel"); break;
        }
        Img0.Adress[2+y*width+x]=(unsigned short) BmpAdress[54+pixel*4+2];
        Img1.Adress[2+y*width+x]=(unsigned short) BmpAdress[54+pixel*4+1];
        Img2.Adress[2+y*width+x]=(unsigned short) BmpAdress[54+pixel*4+0];
      }
    }
  }
}

//**** rgb2bmp ****
void rgb2bmp(name) char *name; {
  unsigned int width,realbmpwidth,height,width0,height0,width1,height1,width2,height2,nb_bpp;
  unsigned int FileLength;
  unsigned int Colors[258],CurrentColor;
  unsigned short red,green,blue;
  int NbColors,ColorFound,ColorIndex;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  width2=Img2.Adress[0];
  height2=Img2.Adress[1];
  //Get common size
  width=width0;
  if(width1<width) { width=width1; }
  if(width2<width) { width=width2; }
  height=height0;
  if(height1<height) { height=height1; }
  if(height2<height) { height=height2; }
  printf("%d x %d",width,height);
  
  //Count number of colors used and put them in palette.
  //Colors are stored in r+g+b;g;b format instead of r;g;b
  //so that we can sort them by intensity.
  NbColors=0;
  for(y=0;y<height;y++) {
    for(x=0;x<width;x++) {
      red=Img0.Adress[2+x+y*width0] &255; Img0.Adress[2+x+y*width0]=red;
      green=Img1.Adress[2+x+y*width1] &255; Img1.Adress[2+x+y*width1]=green;
      blue=Img2.Adress[2+x+y*width2] &255; Img2.Adress[2+x+y*width2]=blue;
      CurrentColor=(unsigned int) ((red+green+blue)<<16)+(green<<8)+blue;
      ColorFound=FALSE;
      ColorIndex=0;
      while((ColorIndex<NbColors) && (ColorFound==FALSE)) {
        if(CurrentColor==Colors[ColorIndex]) { ColorFound=TRUE; }
        ColorIndex++;
      }
      if(ColorFound==FALSE) {
        Colors[NbColors]=CurrentColor;
        if(NbColors!=257) { NbColors++; }
      }
    }
  }
  if(NbColors>256) {
    printf(", >256 colors");
  } else {
    printf(", %d colors",NbColors);
  }
    
  //Sort colors by intensity (r+g+b)
  for(x=0;x<NbColors;x++) {
    //search color with lowest intensity
    ColorIndex=x;
    CurrentColor=Colors[x];
    for(y=x+1;y<NbColors;y++) {
      if(Colors[y]<CurrentColor) {
        CurrentColor=Colors[y];
        ColorIndex=y;
      }
    }
    //exchange with color at x position and put it in r;g;b format
    Colors[ColorIndex]=Colors[x];
    blue=CurrentColor & 255;
    green=CurrentColor & (255*256);
    Colors[x]=CurrentColor-(blue<<16)-(green<<8);
  }
  
  //prepare header
  BmpAdress=(unsigned char *) malloc(54+1024+(width+3)*height*3); // allocate loads of memory
  for(x=0;x<54;x++) { BmpAdress[x]=BmpHeader[x]; } // copy header
  for(x=54;x<54+1024+(width+3)*height*3;x++) { BmpAdress[x]=0; } // fill with 0s
  WriteLongWord(width,18,BmpAdress); // width
  WriteLongWord(height,22,BmpAdress); // height
  
  
  if(NbColors>256) {
    //**** 24bpp image
    realbmpwidth=width*3;
    while((realbmpwidth & 3)!=0) { realbmpwidth++; }
    BmpLength=54+realbmpwidth*height;
    //number of bpp is already 24
    //offset is already 54
    WriteLongWord(BmpLength,2,BmpAdress); // size of file
    for(y=0;y<height;y++) {
      for(x=0;x<width;x++) {
        BmpAdress[54+x*3+y*realbmpwidth+0]=Img2.Adress[2+x+y*width2]; /* blue */
        BmpAdress[54+x*3+y*realbmpwidth+1]=Img1.Adress[2+x+y*width1]; /* green */
        BmpAdress[54+x*3+y*realbmpwidth+2]=Img0.Adress[2+x+y*width0]; /* red */
      }
    }
  } else {
    //**** 8;4 or 1 bpp image with palette
    //copy palette
    for(x=0;x<NbColors;x++) {
      BmpAdress[54+x*4+0]=Colors[x] & 255;
      BmpAdress[54+x*4+1]=(Colors[x]>>8) & 255;
      BmpAdress[54+x*4+2]=(Colors[x]>>16) & 255;
    }
    if(NbColors<3) {
      nb_bpp=1;
    } else if (NbColors<17) {
      nb_bpp=4;
    } else {
      nb_bpp=8;
    }
    
    realbmpwidth=width;
    while(((realbmpwidth*nb_bpp) & 31)!=0) { realbmpwidth++; }
    BmpLength=54+power2(nb_bpp)*4+realbmpwidth*nb_bpp/8*height;
    WriteWord(nb_bpp,28,BmpAdress); //number of bpp
    WriteLongWord(54+power2(nb_bpp)*4,10,BmpAdress); //offset
    WriteLongWord(BmpLength,2,BmpAdress); // size of file
    // copy pixels as palette index
    for(y=0;y<height;y++) {
      for(x=0;x<width;x++) {
        CurrentColor=(unsigned int) (Img0.Adress[2+x+y*width0]<<16)+(Img1.Adress[2+x+y*width1]<<8)+Img2.Adress[2+x+y*width2];
        ColorFound=FALSE;
        ColorIndex=0;
        while(ColorFound==FALSE) {
          if(CurrentColor==Colors[ColorIndex]) {
            ColorFound=TRUE;
            switch(nb_bpp) {
              case 1:BmpAdress[54+2*4+y*realbmpwidth/8+x/8]=BmpAdress[54+2*4+y*realbmpwidth/8+x/8]+(ColorIndex<<(7-x%8)); break;
              case 4:BmpAdress[54+16*4+y*realbmpwidth/2+x/2]=BmpAdress[54+16*4+y*realbmpwidth/2+x/2]+(ColorIndex<<(4*(1-x%2))); break;  
              case 8:BmpAdress[54+256*4+x+y*realbmpwidth]=ColorIndex; break;
            }
          }
          ColorIndex++;
        }
      }
    }
  }
  //save file
  filep=fopen(name,"wb"); /* write binary */
  FileLength=fwrite(BmpAdress,1,BmpLength,filep);
  fclose(filep);
  if(FileLength!=BmpLength) { ErrorInFile(name,"not saved"); }
}

//**** rgb2grey ****
void rgb2grey() {
  unsigned int width,height,width0,height0,width1,height1,width2,height2;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  width2=Img2.Adress[0];
  height2=Img2.Adress[1];
  /* get common size */
  width=width0;
  if(width1<width) { width=width1; }
  if(width2<width) { width=width2; }
  height=height0;
  if(height1<height) { height=height1; }
  if(height2<height) { height=height2; }
  for(y=0;y<height;y++) {
    for(x=0;x<width;x++) {
      t=Img0.Adress[2+x+y*width0]+Img1.Adress[2+x+y*width1]+Img2.Adress[2+x+y*width2]; /* red+green+blue */
      t=divide(t,3);
      Img0.Adress[2+x+y*width]=t;
    }
  }
}

//**** negative ****
void negative() {
  unsigned int c,width0,height0;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  for(c=0;c<width0*height0;c++) {
    Img0.Adress[2+c]=65535-Img0.Adress[2+c];
  }
}

//**** and ****
void and() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;}
  else {width2=width1;}
  if (height0<height1){height2=height0;}
  else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      Img2.Adress[2+width2*y+x]=Img0.Adress[2+width0*y+x]&Img1.Adress[2+width1*y+x];
    }
  }
}

//**** andI ****
void andI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=Img0.Adress[2+width0*y+x]&value;
    }
  }
}

//**** or ****
void or() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;}
  else {width2=width1;}
  if (height0<height1){height2=height0;}
  else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      Img2.Adress[2+width2*y+x]=Img0.Adress[2+width0*y+x]|Img1.Adress[2+width1*y+x];
    }
  }
}

//**** orI ****
void orI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=Img0.Adress[2+width0*y+x]|value;
    }
  }
}

//**** xor ****
void xor() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;}
  else {width2=width1;}
  if (height0<height1){height2=height0;}
  else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      Img2.Adress[2+width2*y+x]=Img0.Adress[2+width0*y+x]^Img1.Adress[2+width1*y+x];
    }
  }
}

//**** xorI ****
void xorI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=Img0.Adress[2+width0*y+x]^value;
    }
  }
}  
  
//**** addborders ****
void addborders(left,right,top,bottom,color) int left,right,bottom,top,color; {
  unsigned int width0,height0,width1,height1;
  int x,y;
  printf("%d %d %d %d",left,right,top,bottom);
  if(left<0) { printf("Error: left should be positive"); exit(0); }
  if(right<0) { printf("Error: right should be positive"); exit(0); }
  if(bottom<0) { printf("Error: bottom should be positive"); exit(0); }
  if(top<0) { printf("Error: top should be positive"); exit(0); }
  if((color<0) || (color>65535)) { printf("Error: color should be in [0;65535]"); exit(0); }
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=left+width0+right;
  height1=bottom+height0+top;
  printf(" -> %d %d",width1,height1);
  Img1=AllocMemForGreyImage(width1,height1);
  for (x=2;x<2+width1*height1;x++) {Img1.Adress[x]=color;}
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img1.Adress[2+(bottom+y)*width1+left+x]=Img0.Adress[2+y*width0+x];
    }
  }
}

//**** removeborders ****
void removeborders(left,right,bottom,top) int left,right,bottom,top; {
  unsigned int width0,height0,width1,height1;
  int x,y;
  printf("%d %d %d %d",left,right,bottom,top);
  if(left<0) { printf("Error: left should be positive"); exit(0); }
  if(right<0) { printf("Error: right should be positive"); exit(0); }
  if(bottom<0) { printf("Error: bottom should be positive"); exit(0); }
  if(top<0) { printf("Error: top should be positive"); exit(0); }
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(left+right>width0) { printf("Error: impossible to remove that many vertical lines"); exit(0); }
  if(top+bottom>height0) { printf("Error: impossible to remove that many horizontal lines"); exit(0); }
  width1=width0-(left+right);
  height1=height0-(bottom+top);
  printf(" -> %d %d",width1,height1);
  Img1=AllocMemForGreyImage(width1,height1);
  for (y=0;y<height1;y++) {
    for (x=0;x<width1;x++) {
      Img1.Adress[2+y*width1+x]=Img0.Adress[2+(y+top)*width0+x+left];
    }
  }
}

//**** autosetborders ****
int GetLineColor(x,y,incx,incy) int x,y,incx,incy; {
  unsigned int width0,height0;
  int color;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  color=(int) Img0.Adress[2+y*width0+x];
  while((x<width0) && (y<height0) && (x>=0) && (y>=0) && (color!=-1)) {
    if(((int) Img0.Adress[2+y*width0+x])!=color) { color=-1; }
    x+=incx;
    y+=incy;
  }
  return(color);  
}  
  
void autosetborders(W) int W; {
  unsigned int width0,height0;
  int x,y;
  int NbUp,NbDown,NbLeft,NbRight; //Nb of lines to remove
  int CurrentBorderColor,BorderColor;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  //count lines to remove up
  NbUp=0;
  BorderColor=GetLineColor(0,0,1,0);
  if(BorderColor!=-1) {
    do{
      NbUp++;
    } while((GetLineColor(0,NbUp,1,0)==BorderColor) && (NbUp<height0));
  }
  //count lines to remove down
  NbDown=0;
  CurrentBorderColor=GetLineColor(0,height0-1,1,0);
  if(CurrentBorderColor!=-1) {
    if((BorderColor!=-1) && (BorderColor!=CurrentBorderColor)) {
      printf("Error: unable to recognize border color"); exit(0);
    }
    BorderColor=CurrentBorderColor;
    do{
      NbDown++;
    } while((GetLineColor(0,height0-1-NbDown,1,0)==BorderColor) && (NbDown<height0));
  }
  //count lines to remove left
  NbLeft=0;
  CurrentBorderColor=GetLineColor(0,0,0,1);
  if(CurrentBorderColor!=-1) {
    if((BorderColor!=-1) && (BorderColor!=CurrentBorderColor)) {
      printf("Error: unable to recognize border color"); exit(0);
    }
    BorderColor=CurrentBorderColor;
    do{
      NbLeft++;
    } while((GetLineColor(NbLeft,0,0,1)==BorderColor) && (NbLeft<width0));
  }
  //count lines to remove left
  NbRight=0;
  CurrentBorderColor=GetLineColor(width0-1,0,0,1);
  if(CurrentBorderColor!=-1) {
    if((BorderColor!=-1) && (BorderColor!=CurrentBorderColor)) {
      printf("Error: unable to recognize border color"); exit(0);
    }
    BorderColor=CurrentBorderColor;
    do{
      NbRight++;
    } while((GetLineColor(width0-1-NbRight,0,0,1)==BorderColor) && (NbRight<width0));
  }
  removeborders(NbLeft,NbRight,NbDown,NbUp);
  //copy Img1 to Img0, possible since Img1 is smaller.
  Img0.Adress[0]=Img1.Adress[0];
  Img0.Adress[1]=Img1.Adress[1];
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img0.Adress[2+y*width0+x]=Img1.Adress[2+y*width0+x];
    }
  }
  printf(" w=%d ",W);
  if(W>0) { addborders(W,W,W,W,BorderColor); }
}

//**** add ****
void add() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;} else {width2=width1;}
  if (height0<height1){height2=height0;} else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      t=Img0.Adress[2+width0*y+x]+Img1.Adress[2+width1*y+x];
      if (t>65535) {
        Img2.Adress[2+width2*y+x]=65535;
      } else {
        Img2.Adress[2+width2*y+x]=t;
      }
    }
  }
}

//**** addI ****
void addI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      t=Img0.Adress[2+width0*y+x]+value;
      if(t>65535) { t=65535; }
      Img2.Adress[2+width0*y+x]=t;
    }
  }
} 

//**** sub ****
void sub() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;} else {width2=width1;}
  if (height0<height1){height2=height0;} else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      t=Img0.Adress[2+width0*y+x]-Img1.Adress[2+width1*y+x];
      if (t<0) {
        Img2.Adress[2+width2*y+x]=0;
      } else {
        Img2.Adress[2+width2*y+x]=t;
      }
    }
  }
} 

//**** subI ****
void subI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      t=Img0.Adress[2+width0*y+x]-value;
      if(t<0) { t=0; }
      Img2.Adress[2+width0*y+x]=t;
    }
  }
} 

//**** rsbI ****
void rsbI(value) unsigned int value; {
  unsigned int width0,height0;
  int x,y,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      t=value-Img0.Adress[2+width0*y+x];
      if(t<0) { t=0; }
      Img2.Adress[2+width0*y+x]=t;
    }
  }
} 

//**** mul ****
void mul() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;} else {width2=width1;}
  if (height0<height1){height2=height0;} else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      Img2.Adress[2+width2*y+x]=Img0.Adress[2+width0*y+x]*Img1.Adress[2+width1*y+x];
    }
  }
} 

//**** mulI ****
void mulI(factor) unsigned int factor; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=Img0.Adress[2+width0*y+x]*factor;
    }
  }
} 

//**** div ****
void division() {
  unsigned int width0,height0,width1,height1,width2,height2;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=Img1.Adress[0];
  height1=Img1.Adress[1];
  if (width0<width1){width2=width0;} else {width2=width1;}
  if (height0<height1){height2=height0;} else {height2=height1;}
  Img2=AllocMemForGreyImage(width2,height2);
  for (y=0;y<height2;y++) {
    for (x=0;x<width2;x++) {
      Img2.Adress[2+width2*y+x]=divide(Img0.Adress[2+width0*y+x],Img1.Adress[2+width1*y+x]);
    }
  }
} 

//**** divI ****
void divI(divisor) unsigned int divisor; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=divide(Img0.Adress[2+width0*y+x],divisor);
    }
  }
} 

//**** rdvI ****
void rdvI(dividend) unsigned int dividend; {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+width0*y+x]=divide(dividend,Img0.Adress[2+width0*y+x]);
    }
  }
} 



//**** filter ****
/* IN  Img0 */
/* OUT Img1 */
void filter(a,b,c,d,e,f,g,h,i,n,offset) int a,b,c,d,e,f,g,h,i,n,offset; {
  unsigned int width,height;
  int x0,x1,x2,y,y0,y1,y2,pixel;
  unsigned short *Adress;
  width=Img0.Adress[0];
  height=Img0.Adress[1];
  Adress=Img0.Adress+2;
  Img1=AllocMemForGreyImage(width,height);
  for(y=0;y<height;y++) {
    y0=((y+height-1)%height)*width; /* if y-1 then modulo doesn't act the intended way */
    y1=y*width;
    y2=((y+1)%height)*width;
    for(x1=0;x1<width;x1++) {
      x0=(x1+width-1)%width;
      x2=(x1+1)%width;
      pixel=a*Adress[y0+x0]+b*Adress[y0+x1]+c*Adress[y0+x2];
      pixel+=d*Adress[y1+x0]+e*Adress[y1+x1]+f*Adress[y1+x2];
      pixel+=g*Adress[y2+x0]+h*Adress[y2+x1]+i*Adress[y2+x2];
      pixel=divide(pixel,n)+offset;
      if(pixel<0) { pixel=0; }
      if(pixel>65535) { pixel=65535; }
      Img1.Adress[2+y1+x1]=pixel;
    }
  }
}

//***** hmirror *****
void hmirror() {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+(height0-1-y)*width0+x]=Img0.Adress[2+y*width0+x];
    }
  }
}

//***** vmirror *****
void vmirror() {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+y*width0+width0-1-x]=Img0.Adress[2+y*width0+x];
    }
  }
}

//***** rotate 90 *****
void rotate90() {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(height0,width0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+(width0-1-x)*height0+y]=Img0.Adress[2+y*width0+x];
    }
  } 
}

//***** rotate180 *****
void rotate180() {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+(height0-1-y)*width0+width0-1-x]=Img0.Adress[2+y*width0+x];
    }
  } 
}

//***** rotate270 *****
void rotate270() {
  unsigned int width0,height0;
  int x,y;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img2=AllocMemForGreyImage(height0,width0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img2.Adress[2+x*height0+height0-1-y]=Img0.Adress[2+y*width0+x];
    }
  } 
}

//**** edge ****
void edge(size,back_color,edge_color) unsigned int size,back_color,edge_color; {
  unsigned int width0,height0,dist,dist2,size2;
  int x,y,yy,found;
  size2=size*size;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img1=AllocMemForGreyImage(height0,width0); /* hdist2 */
  /* Compute and store squares of horizontal distances */
  for(y=0;y<height0;y++) {
    /* from left to right */    
    dist=size;
    dist2=size2;
    for(x=0;x<width0;x++) {
      if(Img0.Adress[2+y*width0+x]==back_color) {
        if(dist<255) { dist2+=2*dist+1; } /* (x+1)^2=x^2+2x+1 */
        dist+=1; /* we won't increase dist2 if dist>=254 */
      } else {
        dist2=0;
        dist=0;
      }
      Img1.Adress[y*width0+x]=dist2;
    }
    /* from right to left */    
    dist=size;
    dist2=size2;
    for(x=width0-1;x>=0;x--) {
      if(Img0.Adress[2+y*width0+x]==back_color) {
        if(dist<255) { dist2+=2*dist+1; }
        dist+=1;
      } else {
        dist2=0;
        dist=0;
      }
      if(dist2<Img1.Adress[y*width0+x]) { Img1.Adress[y*width0+x]=dist2; }
    }
  }
  /* Compute dist2 using previous table and Pythagora's theorem */
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      if(Img0.Adress[2+y*width0+x]==back_color) {
        found=FALSE;
        dist=0;
        dist2=0;
        while((dist<size) && (found==FALSE)) {
          yy=y+dist;
          if((yy<height0) && (dist2+Img1.Adress[yy*width0+x]<size2)) { found=TRUE; }
          yy=y-dist;
          if((yy>=0) && (dist2+Img1.Adress[yy*width0+x]<size2)) { found=TRUE; }
          if(dist<255) { dist2+=2*dist+1; }
          dist++;
        }
        if(found==TRUE) { Img0.Adress[2+y*width0+x]=edge_color; }
      }
    }
  }
}

//***** explode *****
void explode(NbIt) unsigned int NbIt; {
  unsigned int width0,height0,x,y,c,t;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  for(c=0;c<NbIt;c++) {
    x=rnd(0,width0-2);
    y=rnd(0,height0-2);
    t=Img0.Adress[2+y*width0+x];
    if(rnd(0,1)==0) {
      Img0.Adress[2+y*width0+x]=Img0.Adress[2+y*width0+x+1];
      Img0.Adress[2+y*width0+x+1]=t;
    } else {
      Img0.Adress[2+y*width0+x]=Img0.Adress[2+(y+1)*width0+x];
      Img0.Adress[2+(y+1)*width0+x]=t;
    }
  }
}

//**** palette2rgb ****
void palette2rgb(name) char *name; {
  unsigned int width0,height0,NbColors;
  int x,y,PaletteLength;
  unsigned short t;
  unsigned char *Palette;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img1=AllocMemForGreyImage(width0,height0);
  Img2=AllocMemForGreyImage(width0,height0);
  filep=fopen(name,"rb"); // rb = read binary
  if(filep==0) { ErrorInFile(name,"not found"); }
  fseek(filep,0,2); // set the file pointer at the end of file
  PaletteLength=(int) ftell(filep); // returns the length of the file in bytes
  fseek(filep,0,0); // set the file pointer at the start of file
  Palette=(unsigned char *) malloc(PaletteLength); // allocate memory
  fread(Palette,1,PaletteLength,filep); // load file, 1=size of a byte
  fclose(filep);
  NbColors=PaletteLength/3;
  printf("%d colors",NbColors);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      t=Img0.Adress[2+y*width0+x];
      if(t>=NbColors) { t=NbColors-1; }
      Img0.Adress[2+y*width0+x]=Palette[t*3];
      Img1.Adress[2+y*width0+x]=Palette[t*3+1];
      Img2.Adress[2+y*width0+x]=Palette[t*3+2];
    }
  }
}

//**** textpalette2rgb ****
void textpalette2rgb(name) char *name; {
  unsigned int width0,height0;
  int x,y,PaletteLength,NbValues,NbNibbles,NbColors;
  unsigned short t;
  unsigned char *Palette;
  unsigned char c,Value;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  Img1=AllocMemForGreyImage(width0,height0);
  Img2=AllocMemForGreyImage(width0,height0);
  filep=fopen(name,"rb"); // rb = read binary
  if(filep==0) { ErrorInFile(name,"not found"); }
  fseek(filep,0,2); // set the file pointer at the end of file
  PaletteLength=(int) ftell(filep); // returns the length of the file in bytes
  fseek(filep,0,0); // set the file pointer at the start of file
  Palette=(unsigned char *) malloc(PaletteLength); // allocate memory
  fread(Palette,1,PaletteLength+1,filep); // load file, 1=size of a byte
  Palette[PaletteLength]=10; //append EOL ($0a)
  fclose(filep);
  //convert textpalette
  NbValues=0;
  Value=0;
  NbNibbles=0;
  x=0;
  while(x<PaletteLength) {
    c=Palette[x];
    if(c==';') {
      while(c!=10) { x++; c=Palette[x]; } // skip comment till EOL ($0a)
    } else {
      if(c>='a') { c=c-32; } //convert to lowercase
      if(((c>='0') && (c<='9')) || ((c>='A') && (c<='F'))) {
        if(c>='A') { c=c-'A'+10; } else { c=c-'0'; }
        Value=Value*16+c;
        NbNibbles++;
        if(NbNibbles==2) {
          Palette[NbValues]=Value;
          NbValues++;
          NbNibbles=0;
          Value=0;
        }
      }
    }
    x++;
  }
  NbColors=NbValues/3;
  
//  for(x=0;x<NbColors;x++) {
//    printf("%x ",Palette[x]);
//  }
  
  printf("%d colors",NbColors);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      t=Img0.Adress[2+y*width0+x];
      if(t>=NbColors) { t=NbColors-1; }
      Img0.Adress[2+y*width0+x]=Palette[t*3];
      Img1.Adress[2+y*width0+x]=Palette[t*3+1];
      Img2.Adress[2+y*width0+x]=Palette[t*3+2];
    }
  }
}


//**** setmincolor ****
// substract oldmincolor-newmincolor to every pixel
void setmincolor(newmincolor) unsigned short newmincolor; {
  int x,y;
  unsigned short width0,height0,oldmincolor;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  oldmincolor=searchmincolorinImg0();
  printf("oldmincolor=%d",oldmincolor);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      Img0.Adress[2+y*width0+x]=Img0.Adress[2+y*width0+x]-oldmincolor+newmincolor;
    }
  }
}

//**** setmaxcolor ****
// substract oldmaxcolor-newmaxcolor to every pixel
void setmaxcolor(newmaxcolor) unsigned short newmaxcolor; {
  int x,y;
  unsigned short width0,height0,oldmaxcolor;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  oldmaxcolor=searchmaxcolorinImg0();
  printf("oldmaxcolor=%d",oldmaxcolor);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      Img0.Adress[2+y*width0+x]=Img0.Adress[2+y*width0+x]-oldmaxcolor+newmaxcolor;
    }
  }
}

//**** allcolorsabove ****
// if color<mincolor then color=mincolor
void allcolorsabove(mincol) unsigned short mincol; {
  int x,y;
  unsigned short width0,height0,color;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  printf("oldmincolor=%d",searchmincolorinImg0());
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x];
      if(color<mincol) { Img0.Adress[2+y*width0+x]=mincol; }
    }
  }
}

//**** allcolorsbelow ****
// if color>maxcolor then color=maxcolor
void allcolorsbelow(maxcol) unsigned short maxcol; {
  int x,y;
  unsigned short width0,height0,color;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  printf("oldmaxcolor=%d",searchmaxcolorinImg0());
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x];
      if(color>maxcol) { Img0.Adress[2+y*width0+x]=maxcol; }
    }
  }
}

//**** allcolorsbetween ****
// if color<mincolor then color=mincolor, etc
void allcolorsbetween(colmin,colmax) unsigned short colmin,colmax; {
  int x,y;
  unsigned short width0,height0,col;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  printf("colors previously in [%d;%d]",searchmincolorinImg0(),searchmaxcolorinImg0());
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      col=Img0.Adress[2+y*width0+x];
      if(col<colmin) { Img0.Adress[2+y*width0+x]=colmin; }
      if(col>colmax) { Img0.Adress[2+y*width0+x]=colmax; }
    }
  }
}

//**** mknoise ****
void mknoise(min,max,width0,height0) unsigned short min,max; unsigned int width0,height0; {
  int x,y;
  Img0=AllocMemForGreyImage(width0,height0);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      Img0.Adress[2+y*width0+x]=rnd(min,max);
    }
  }  
}

//**** disc ****
/* returns nb of pixels in disc */
int disc(cx,cy,radius,col) unsigned short cx,cy,radius,col; {
  int x,y,dx,dy,dist2,dist2max;
  int NbOfPixels=0;
  unsigned short width0,height0;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(radius!=0) {
    // print first line
    //!!!SHOULD BE if((cy>=0) && (cy<height0)) { BUT CY>=0 ALWAYS TRUE
    if(cy<height0) {
      for(x=cx-radius;x<=cx+radius;x++) {
        if((x>=0) && (x<width0)) { Img0.Adress[2+cy*width0+x]=col; }
      }
    }
    NbOfPixels+=2*radius+1;
    // now process two lines at a time
    dx=radius;
    dy=1;
    dist2=radius*radius;
    dist2max=dist2+radius;
    dist2+=1;
    while(dy<=radius) {
      while(dist2>dist2max) {
        dist2+=-2*dx+1; // dist2=(dx-1)^2+dy^2=old_dist2-2*dx+1
        dx-=1;
      }
      // draw two lines
      y=cy+dy;
      if((y>=0) && (y<height0)) {
        y=y*width0;
        for(x=cx-dx;x<=cx+dx;x++) {
          if((x>=0) && (x<width0)) { Img0.Adress[2+y+x]=col; }
        }
      }
      y=cy-dy;
      if((y>=0) && (y<height0)) {
        y=y*width0;
        for(x=cx-dx;x<=cx+dx;x++) {
          if((x>=0) && (x<width0)) { Img0.Adress[2+y+x]=col; }
        }
      }
      NbOfPixels+=2*(2*dx+1);
      dist2+=2*dy+1; // dist2=dx^2+(dy+1)^2=old_dist2+2*dy+1
      dy+=1;
    }
  }
  return(NbOfPixels);
}

//**** rectangle ****
void rectangle(xleft,ybottom,xright,ytop,color) unsigned int xleft,ybottom,xright,ytop,color; {
  int x,y;
  unsigned short width0,height0;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(xleft<0) { xleft=0; }
  if(xright<0) { xright=0; }
  if(xleft>xright) { printf("Error: xleft should be smaller than xright"); exit(0); }
  if(ybottom<0) { ybottom=0; }
  if(ytop<0) { ytop=0; }
  if(ybottom>ytop) { printf("Error: ybottom should be smaller than ytop"); exit(0); }
  if((color<0) || (color>65535)) { printf("Error: color should be in [0;65535]"); exit(0); }
  if (xright>=width0) { xright=width0-1; }
  if (ytop>=height0) { ytop=height0-1; }
  printf("(%d;%d) to (%d;%d)",xleft,ybottom,xright,ytop);
  for (y=ybottom;y<=ytop;y++) {
    for (x=xleft;x<=xright;x++) {
      Img0.Adress[2+y*width0+x]=color;
    }
  }
}

//**** rectangle2 ****
void rectangle2(xleft,ybottom,width,height,color) unsigned int xleft,ybottom,width,height,color; {
  rectangle(xleft,ybottom,xleft+width-1,ybottom+height-1);
}

//**** landscape ****
void landscape(nbiter) int nbiter; {
  int x,y,xx,yy,t,i,fractalfactor;
  unsigned short w,h;
  unsigned short a,b,c,d;
  unsigned short *temp;
  w=Img0.Adress[0];
  h=Img0.Adress[1];
  t=power2(nbiter);
  Img1=AllocMemForGreyImage(w*t,h*t);
  Img2=AllocMemForGreyImage(w*t,h*t);
  
  for(x=2;x<2+w*h;x++) {Img1.Adress[x]=Img0.Adress[x];}
  for(i=0;i<nbiter;i++) {
    fractalfactor=power2(nbiter-i);
    for(y=0;y<h;y++) {
      yy=y+1; if(yy==h) { yy=0; }
      for(x=0;x<w;x++) {
        xx=x+1; if(xx==w) { xx=0; }
        a=Img1.Adress[2+y*w+x];
        b=Img1.Adress[2+y*w+xx];
        c=Img1.Adress[2+yy*w+x];
        d=Img1.Adress[2+yy*w+xx];
        Img2.Adress[2+2*(y*2*w+x)]=a;
        t=(a+b)/2+rnd(-fractalfactor,fractalfactor);
        if(t<0) { t=0;} if(t>255) { t=255; }
        Img2.Adress[2+2*(y*2*w+x)+1]=t;
        t=(a+c)/2+rnd(-fractalfactor,fractalfactor);
        if(t<0) { t=0;} if(t>255) { t=255; }
        Img2.Adress[2+2*(y*2*w+x)+2*w]=t;
        t=(a+b+c+d)/4+rnd(-fractalfactor,fractalfactor);
        if(t<0) { t=0;} if(t>255) { t=255; }
        Img2.Adress[2+2*(y*2*w+x)+2*w+1]=t;
      }
    }
    temp=Img2.Adress;
    Img2.Adress=Img1.Adress;
    Img1.Adress=temp;
    w=2*w;
    h=2*h;
  }
}


//**** range ****
void range(colmin,colmax) unsigned short colmin,colmax; {
  int x,y;
  unsigned short width0,height0,col;
  unsigned short currentmincol,currentmaxcol;
  int delta;
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  delta=currentmaxcol-currentmincol;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      col=Img0.Adress[2+y*width0+x];
      Img0.Adress[2+y*width0+x]=colmin+divide((colmax-colmin)*(col-currentmincol),delta);
    }
  }
}


//**** fillwithOs ****
unsigned short FWC_width0,FWC_height0;
int FWC_cx,FWC_cy,FWC_radius;

// returns -1 if circle contains one (or more) non empty pixel
int FWC_CountEmptyPixelsInCircle() {
  int x,y1,y2,dx,dy,dist2,dist2max;
  int NbPixelsInDisc=0;
  // first line
  y1=FWC_cy*FWC_width0;
  for(x=FWC_cx-FWC_radius;x<=FWC_cx+FWC_radius;x++) {
    if(Img0.Adress[2+y1+x]!=0) { return(-1); }
  }
  NbPixelsInDisc+=2*FWC_radius+1;
  // now process two lines at a time
  dx=FWC_radius;
  dy=1;
  dist2=FWC_radius*FWC_radius;
  dist2max=dist2+FWC_radius;
  dist2+=1;
  while(dy<=FWC_radius) {
    while(dist2>dist2max) {
      dist2+=-2*dx+1; // dist2=(dx-1)^2+dy^2=old_dist2-2*dx+1
      dx-=1;
    }
    // check both hlines
    y1=(FWC_cy+dy)*FWC_width0;
    y2=(FWC_cy-dy)*FWC_width0;
    for(x=FWC_cx-FWC_radius;x<=FWC_cx+FWC_radius;x++) {
      if(Img0.Adress[2+y1+x]!=0) { return(-1); }
      if(Img0.Adress[2+y2+x]!=0) { return(-1); }
    }
    NbPixelsInDisc+=2*(2*dx+1);
    dist2+=2*dy+1; // dist2=dx^2+(dy+1)^2=old_dist2+2*dy+1
    dy+=1;
  }
  return(NbPixelsInDisc);
}

void fillwithOs(OutRMin,OutRMax,InRMin,InRMax,Border,ClearColor,CircleColor)
  unsigned short OutRMin,OutRMax,InRMin,InRMax,Border,ClearColor,CircleColor; {
    
  unsigned int c,NbFruitlessTries;
  int NbEmptyPixels;
  int x,y;
  FWC_width0=Img0.Adress[0];
  FWC_height0=Img0.Adress[1];
  // Count the number of empty pixels
  NbEmptyPixels=0;
  for(y=0;y<FWC_height0;y++) {
    for(x=0;x<FWC_width0;x++) {
      if(Img0.Adress[2+y*FWC_width0+x]==0) { NbEmptyPixels++; }
    }
  }
  
  // As long as we can, try to add a circle in black parts
  NbFruitlessTries=0;
  while((NbFruitlessTries<1000) && (NbEmptyPixels>0)) {
    // Choose the center amongst empty pixels
    c=rnd(1,NbEmptyPixels);
    FWC_cx=-1; FWC_cy=0;
    do {
      do {
        FWC_cx++;
        if(FWC_cx==FWC_width0) {
          FWC_cx=0;
          FWC_cy+=1;
        }
      } while(Img0.Adress[2+FWC_cy*FWC_width0+FWC_cx]!=0);
      c-=1;
    } while(c>0);
    // Here FWC_cx,FWC_cy is the center of the circle
    
    // Set up radius as rmax at first, clip to borders
    FWC_radius=OutRMax+Border;
    if(FWC_cx-FWC_radius<0) { FWC_radius=FWC_cx; }
    if(FWC_cx+FWC_radius>FWC_width0-1) { FWC_radius=FWC_width0-1-FWC_cx; }
    if(FWC_cy-FWC_radius<0) { FWC_radius=FWC_cy; }
    if(FWC_cy+FWC_radius>FWC_height0-1) { FWC_radius=FWC_height0-1-FWC_cy; }
    
    // See if (non clipped) circle contains only empty pixels, else decrease radius
    c=FWC_CountEmptyPixelsInCircle();
    while((c==-1) && (FWC_radius>=OutRMin)) {
      FWC_radius-=1;
      c=FWC_CountEmptyPixelsInCircle();
    }
    if(FWC_radius-Border>=OutRMin) {
      NbEmptyPixels-=c; // remove pixels from all circles
      c=disc(FWC_cx,FWC_cy,FWC_radius,ClearColor); // clear
      c-=disc(FWC_cx,FWC_cy,FWC_radius-Border,CircleColor); // draw circle
      if(FWC_radius-Border-1>=InRMin) {
        x=InRMax;
        if(FWC_radius-Border-1<x) { x=FWC_radius-Border-1; }
        FWC_radius=rnd(InRMin,x);
        c+=disc(FWC_cx,FWC_cy,FWC_radius,ClearColor); // clear inside
      }
      if(ClearColor==0) { NbEmptyPixels+=c; } // pixels==0 can be accessed again
      NbFruitlessTries=0;
    } else {
      NbFruitlessTries+=1;
    }
  }
}

//**** floyd-steinberg ****
//error propagated to other pixels
//    . 7
//  3 5 1
void floydsteinberg(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,col,newcol;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,newerror,nbcols;
  float errorcurrentline[4096];
  float errornextline[4096];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if((width0>4096) || (height0>4096)) { ErrorInFile("image","too big"); }
  for(x=0;x<width0;x++) { errornextline[x]=0.0; }
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      errorcurrentline[x]=errornextline[x];
      errornextline[x]=0.0;
    }
    for(x=0;x<width0;x++) {
      col=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) col)/amplitude;
      intensity+=errorcurrentline[x];
      if(intensity>1.0) { intensity=1.0; }
      if(intensity<0.0) { intensity=0.0; }
      newcol=(unsigned short) (intensity*(nbcols-1));
      newerror=intensity- ((float) newcol)/nbcols;
//!!!      newerror=intensity-((float) newcol)/(nbcols-1.0);
//!!!      newerror=intensity-(0.5+(float) newcol)/nbcols;
      Img0.Adress[2+y*width0+x]=newcol;
      // propagate error
      if(x!=width0-1) {
        errorcurrentline[x+1]+=7*newerror/16;
        errornextline[x+1]+=1*newerror/16;
      }
      errornextline[x]+=5*newerror/16;
      if(x!=0) { errornextline[x-1]+=3*newerror/16; }
    }
  }
}

//**** stucki ****
//error propagated to other pixels
//      . 8 4
//  2 4 8 4 2
//  1 2 4 2 1
void stucki(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,col,newcol;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,newerror,nbcols;
  float errorcurrentline[4096];
  float errornextline[4096];
  float errornextline2[4096];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if((width0>4096) || (height0>4096)) { ErrorInFile("image","too big"); }
  for(x=0;x<width0;x++) { errornextline[x]=0.0; errornextline2[x]=0.0; }
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      errorcurrentline[x]=errornextline[x];
      errornextline[x]=errornextline2[x];
      errornextline2[x]=0.0;
    }
    for(x=0;x<width0;x++) {
      col=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) col)/amplitude;
      intensity+=errorcurrentline[x];
      if(intensity>1.0) { intensity=1.0; }
      if(intensity<0.0) { intensity=0.0; }
      newcol=(unsigned short) (intensity*(nbcols-1));
      newerror=intensity- ((float) newcol)/nbcols;
//!!!      newerror=intensity-((float) newcol)/(nbcols-1.0);
//!!!      newerror=intensity-(0.5+(float) newcol)/nbcols;
      Img0.Adress[2+y*width0+x]=newcol;
      // propagate error
      if(x<width0-2) {
        errorcurrentline[x+2]+=4*newerror/42;
        errornextline[x+2]+=2*newerror/42;
        errornextline2[x+2]+=1*newerror/42;
      }
      if(x<width0-1) {
        errorcurrentline[x+1]+=8*newerror/42;
        errornextline[x+1]+=4*newerror/42;
        errornextline2[x+1]+=2*newerror/42;
      }
      errornextline[x]+=8*newerror/42;
      errornextline2[x]+=4*newerror/42;
      if(x>0) {
        errornextline[x-1]+=4*newerror/42;
        errornextline2[x-1]+=2*newerror/42;
      }
      if(x>1) {
        errornextline[x-2]+=2*newerror/42;
        errornextline2[x-2]+=1*newerror/42;
      }
    }
  }
}

//**** ordered2x1 ****
void ordered1x2(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[2]={1.0/3.0 , 2.0/3.0};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[y%2]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** ordered2x1 ****
void ordered2x1(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[2]={1.0/3.0 , 2.0/3.0};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[x%2]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** ordered2x2 ****
void ordered2x2(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[2][2]={{1.0/5.0 , 3.0/5.0} ,
                         {4.0/5.0 , 2.0/5.0}};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[x%2][y%2]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** ordered3x3 ****
void ordered3x3(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[3][3]={{3.0/10.0 , 7.0/10.0 , 4.0/10.0} ,
                         {6.0/10.0 , 1.0/10.0 , 9.0/10.0} ,
                         {2.0/10.0 , 8.0/10.0 , 5.0/10.0}};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[x%3][y%3]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** ordered4x4 ****
void ordered4x4(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[4][4]={{01.0/17.0 , 09.0/17.0 , 03.0/17.0 , 11.0/17.0} ,
                         {13.0/17.0 , 05.0/17.0 , 15.0/17.0 , 07.0/17.0} ,
                         {04.0/17.0 , 12.0/17.0 , 02.0/17.0 , 10.0/17.0} ,
                         {16.0/17.0 , 08.0/17.0 , 14.0/17.0 , 06.0/17.0}};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[x%4][y%4]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** ordered4x4bis ****
void ordered4x4bis(NbColors) unsigned short NbColors; {
  int x,y;
  unsigned short width0,height0,color,newcolor;
  unsigned short currentmincol,currentmaxcol;
  float amplitude,intensity,nbcols;
  float threshold[4][4]={{01.0/17.0 , 05.0/17.0 , 09.0/17.0 , 13.0/17.0} ,
                         {14.0/17.0 , 02.0/17.0 , 06.0/17.0 , 10.0/17.0} ,
                         {11.0/17.0 , 15.0/17.0 , 03.0/17.0 , 07.0/17.0} ,
                         {08.0/17.0 , 12.0/17.0 , 16.0/17.0 , 04.0/17.0}};
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  nbcols=(float) (NbColors);
  currentmincol=searchmincolorinImg0();
  currentmaxcol=searchmaxcolorinImg0();
  amplitude=(float) (currentmaxcol-currentmincol);
  for(y=0;y<height0;y++) {
    for(x=0;x<width0;x++) {
      color=Img0.Adress[2+y*width0+x]-currentmincol;
      intensity=((float) color)/amplitude;
      newcolor=(unsigned short) (intensity*(nbcols-1)+threshold[x%4][y%4]);
      Img0.Adress[2+y*width0+x]=newcolor;
    }
  }
}

//**** sampleup_miller ****
void sampleup_miller(nbiterations) unsigned int nbiterations; {
  unsigned int width,height,nextwidth;
  int n,x,y,x2,y2,h00,h01,h10,h11;
  width=Img0.Adress[0];
  height=Img0.Adress[1];
  Img1=AllocMemForGreyImage(width*power2(nbiterations),height*power2(nbiterations));
  Img2=AllocMemForGreyImage(width*power2(nbiterations-1),height*power2(nbiterations-1));
  for(x=0;x<2+width*height;x++) { Img1.Adress[x]=Img0.Adress[x]; } //copy Img0 to Img1
  for(n=0;n<nbiterations;n++) {
    width=Img1.Adress[0];
    height=Img1.Adress[1];
    for(x=0;x<2+width*height;x++) { Img2.Adress[x]=Img1.Adress[x]; } //copy Img1 to Img2
    nextwidth=2*width;
    Img1.Adress[0]=nextwidth;
    Img1.Adress[1]=2*height;
    for(y=0;y<height;y++) {
      y2=y+1; if(y2==height) { y2=0; }
      for(x=0;x<width;x++) {
        x2=x+1; if(x2==width) { x2=0; }
        h00=Img2.Adress[2+y*width+x];
        h10=Img2.Adress[2+y*width+x2];
        h01=Img2.Adress[2+y2*width+x];
        h11=Img2.Adress[2+y2*width+x2];
        Img1.Adress[2+2*y*nextwidth+2*x]=divide(h00*9+(h10+h01)*3+h11,16);
        Img1.Adress[2+2*y*nextwidth+(2*x+1)]=divide(h10*9+(h00+h11)*3+h01,16);
        Img1.Adress[2+(2*y+1)*nextwidth+(2*x+1)]=divide(h11*9+(h01+h10)*3+h00,16);
        Img1.Adress[2+(2*y+1)*nextwidth+2*x]=divide(h01*9+(h11+h00)*3+h10,16);
      }
    }
  }
  printf("%d x %d",Img1.Adress[0],Img1.Adress[1]);
}

//**** sampledown ****
void sampledown(xratio,yratio) short xratio,yratio; {
  unsigned int width0,height0,width1,height1;
  int x,y,xsrc,ysrc,xsrc2,ysrc2,nbpixelsinrectangle,sumpixels;
  nbpixelsinrectangle=xratio*yratio;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(width0%xratio!=0) { printf("Error: %d is not a multiple of %d",width0,xratio); exit(0); }
  if(height0%yratio!=0) { printf("Error: %d is not a multiple of %d",height0,yratio); exit(0); }
  width1=width0/xratio;
  height1=height0/yratio;
  Img1=AllocMemForGreyImage(width1,height1);
  for (y=0;y<height1;y++) {
    ysrc=y*yratio;
    for (x=0;x<width1;x++) {
      xsrc=x*xratio;
      sumpixels=0;
      for(ysrc2=ysrc;ysrc2<ysrc+yratio;ysrc2++) {
        for(xsrc2=xsrc;xsrc2<xsrc+xratio;xsrc2++) {
          sumpixels+=Img0.Adress[2+ysrc2*width0+xsrc2];
        }
      }
      Img1.Adress[2+y*width1+x]=sumpixels/nbpixelsinrectangle;
    }
  }
}

//**** scaleup ****
void scaleup(xratio,yratio) short xratio,yratio; {
  unsigned int width0,height0,width1,height1;
  int x,y,xdest,ydest;
  unsigned short pixel;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  width1=width0*xratio;
  height1=height0*yratio;
  Img1=AllocMemForGreyImage(width1,height1);
  for (y=0;y<height0;y++) {
    for (x=0;x<width0;x++) {
      pixel=Img0.Adress[2+y*width0+x];
      for(xdest=x*xratio;xdest<(x+1)*xratio;xdest++) {
        for(ydest=y*yratio;ydest<(y+1)*yratio;ydest++) {
          Img1.Adress[2+ydest*width1+xdest]=pixel;
        }
      }
    }
  }
}

//**** scaledown ****
void scaledown(xratio,yratio) short xratio,yratio; {
  unsigned int width0,height0,width1,height1;
  int x,y,ysrc;
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(width0%xratio!=0) { printf("Error: %d is not a multiple of %d",width0,xratio); exit(0); }
  if(height0%yratio!=0) { printf("Error: %d is not a multiple of %d",height0,yratio); exit(0); }
  width1=width0/xratio;
  height1=height0/yratio;
  Img1=AllocMemForGreyImage(width1,height1);
  for (y=0;y<height1;y++) {
    ysrc=y*yratio;
    for (x=0;x<width1;x++) {
      Img1.Adress[2+y*width1+x]=Img0.Adress[2+ysrc*width0+x*xratio];
    }
  }
}

//**** mkstrippedtube ****
void mkstrippedtube(diameter,nbsteps,nbstripes,col0,col1) short diameter,nbsteps,nbstripes; {
  unsigned int width0,height0;
  int x,y,t;
  double startangle,angle;
  Img0=AllocMemForGreyImage(diameter,nbsteps);
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  printf("%d x %d",width0,height0);
  for(y=0;y<nbsteps;y++) {
    startangle=2*pi*((double) y)/((double) nbstripes*nbsteps);
    for(x=0;x<diameter;x++) {
      angle=acos( ((double) (2*x-diameter))/((double) diameter));
      t=(short) (pi+(angle-startangle)*((double) nbstripes/pi)); //+pi to avoid negative value
      if(t%2==0) { t=col0; } else { t=col1; }
      Img0.Adress[2+y*width0+x]=t;
    }
  }
}

//**** grey2bitplanes ****
void grey2bitplanes(nbbitplanes,name)  short nbbitplanes; char *name; {
  unsigned int FileLength;
  unsigned int width0,height0,nbwords;
  unsigned short int bitplane[nbbitplanes];
  unsigned char *File;
  FILE *filep;
  int x,y,word,pixel,t,FilePos;
  
  width0=Img0.Adress[0];
  height0=Img0.Adress[1];
  if(width0%16!=0) { printf("Error: width=%d is not a multiple of 16",width0); exit(0); }
  nbwords=width0/16;
  FileLength=2*nbbitplanes*nbwords*height0;
  File=(unsigned char *) malloc(FileLength);
  
  FilePos=0;
  for(y=0;y<height0;y++) {
    for(word=0;word<nbwords;word++) {
      for(t=0;t<nbbitplanes;t++) { bitplane[t]=0; }
      for(x=word*16;x<word*16+16;x++) {
        pixel=Img0.Adress[2+y*width0+x];
        for(t=0;t<nbbitplanes;t++) {
          bitplane[t]=2*bitplane[t]+(pixel & 1);
          pixel=pixel>>1;
        }
      }
      for(t=0;t<nbbitplanes;t++) {
        //Big endian vs small
        File[FilePos]=(unsigned char) (bitplane[t]>>8);
        FilePos++;
        File[FilePos]=(unsigned char) (bitplane[t] & 255);
        FilePos++;
      }
    }
  }
  
  filep=fopen(name,"wb"); //write binary
  t=fwrite(File,1,FileLength,filep);
  fclose(filep);
  if(FileLength!=t) { ErrorInFile(name,"not saved"); }
}
  

////**** Save Image ****
//void SaveImage(name,Img) char *name; IMAGE Img; {
//  unsigned int FileLength;
//  filep=fopen(name,"wb"); /* write binary */
//  FileLength=fwrite(Img.Adress,1,Img.Length,filep);
//  fclose(filep);
//  if(FileLength!=Img.Length) { ErrorInFile(name,"not saved"); }
//}
    
    
//**** ReadNumber ****
int ReadNumber(s) char *s; {
  int n,t,sign;
  unsigned char c;
  n=0;
  sign=1;
  t=0;
  c=s[t];
  if(c=='-') { sign=-1; t++; c=s[t]; }
  while(c!=0) {
    if((c<'0') || (c>'9')) {
      printf("Error: %s should be a number",s);
      exit(1);
    }
    n=n*10+c-'0';
    t++;
    c=s[t];
  }
  return(n*sign);
}

//**** MAIN *****************************************************************
int main(argc,argv) int argc; char *argv[]; {
#ifdef ALAIN
  rndX=rawclock(); // initialise random seed
#else  
  rndX=time(0); // initialise random seed
#endif    
//  rndX=1; // initialise random seed

  if(argc<=2) { printf("Usage: fab function parameter ... parameter\nSee AllFunctions.txt"); exit(1); }
  
  if(strcmp(argv[1],"bmp2rgb")==0) {
    CheckUsage(argc,6,"bmp2rgb","in.bmp red.g green.g blue.g");
    bmp2rgb(argv[2]);
    SaveImage(argv[3],Img0);
    SaveImage(argv[4],Img1);
    SaveImage(argv[5],Img2);
    
  } else if (strcmp(argv[1],"rgb2bmp")==0) {
    CheckUsage(argc,6,"rgb2bmp","red.g green.g blue.g out.bmp");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    Img2=AllocMemAndLoadGreyImage(argv[4]);
    rgb2bmp(argv[5]);
    
  } else if (strcmp(argv[1],"rgb2grey")==0) {
    CheckUsage(argc,6,"rgb2grey","red.g green.g blue.g grey.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    Img2=AllocMemAndLoadGreyImage(argv[4]);
    rgb2grey();
    SaveImage(argv[5],Img0);
   
  } else if (strcmp(argv[1],"negative")==0) {
    CheckUsage(argc,4,"negative","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    negative();
    SaveImage(argv[3],Img0);

  } else if (strcmp(argv[1],"filter")==0) {
    CheckUsage(argc,15,"filter","a b c d e f g h i divisor offset in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[13]);
    filter(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]),ReadNumber(argv[7]),ReadNumber(argv[8]),ReadNumber(argv[9]),ReadNumber(argv[10]),ReadNumber(argv[11]),ReadNumber(argv[12]));
    SaveImage(argv[14],Img1);
    
  } else if (strcmp(argv[1],"and")==0) {
    CheckUsage(argc,5,"and","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    and();
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"andI")==0) {
    CheckUsage(argc,5,"andI","value in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    andI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"or")==0) {
    CheckUsage(argc,5,"or","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    or();
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"orI")==0) {
    CheckUsage(argc,5,"orI","value in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    orI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"xor")==0) {
    CheckUsage(argc,5,"xor","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    xor();
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"xorI")==0) {
    CheckUsage(argc,5,"xorI","value in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    xorI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);
      
  } else if (strcmp(argv[1],"addborders")==0) {
    CheckUsage(argc,9,"addborders","left right up down color in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[7]);
    addborders(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]));
    SaveImage(argv[8],Img1);
    
  } else if (strcmp(argv[1],"removeborders")==0) {
    CheckUsage(argc,8,"removeborders","left right up down in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[6]);
    removeborders(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]));
    SaveImage(argv[7],Img1);
    
  } else if (strcmp(argv[1],"autosetborders")==0) {
    CheckUsage(argc,4,"autosetborders","width in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    autosetborders(ReadNumber(argv[2]));
    SaveImage(argv[4],Img1);

  } else if (strcmp(argv[1],"add")==0) {
    CheckUsage(argc,5,"add","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    add();
    SaveImage(argv[4],Img2);    
  
   } else if (strcmp(argv[1],"addI")==0) {
    CheckUsage(argc,5,"addI","value in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    addI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);    
    
   } else if (strcmp(argv[1],"sub")==0) {
    CheckUsage(argc,5,"sub","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    sub();
    SaveImage(argv[4],Img2);    
  
   } else if (strcmp(argv[1],"subI")==0) {
    CheckUsage(argc,5,"subI","in.g value out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    subI(ReadNumber(argv[3]));
    SaveImage(argv[4],Img2);    
    
   } else if (strcmp(argv[1],"rsbI")==0) {
    CheckUsage(argc,5,"rsbI","value in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    rsbI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);    
    
  } else if (strcmp(argv[1],"mul")==0) {
    CheckUsage(argc,5,"mul","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    mul();
    SaveImage(argv[4],Img2);     
    
  } else if (strcmp(argv[1],"mulI")==0) {
    CheckUsage(argc,5,"mulI","factor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    mulI(ReadNumber(argv[2]));
    SaveImage(argv[4],Img2);     
    
  } else if (strcmp(argv[1],"div")==0) {
    CheckUsage(argc,5,"div","in1.g in2.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    Img1=AllocMemAndLoadGreyImage(argv[3]);
    division();
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"divI")==0) {
    CheckUsage(argc,5,"divI","in.g divisor out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    divI(ReadNumber(argv[3]));
    SaveImage(argv[4],Img2);

  } else if (strcmp(argv[1],"rdvI")==0) {
    CheckUsage(argc,5,"rdvI","dividend in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    rdvI(ReadNumber(argv[3]));
    SaveImage(argv[4],Img2);
    
  } else if (strcmp(argv[1],"hmirror")==0) {    
    CheckUsage(argc,4,"hmirror","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    hmirror();
    SaveImage(argv[3],Img2);

  } else if (strcmp(argv[1],"vmirror")==0) {    
    CheckUsage(argc,4,"vmirror","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    vmirror();
    SaveImage(argv[3],Img2);

  } else if (strcmp(argv[1],"rotate90")==0) {    
    CheckUsage(argc,4,"rotate90","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    rotate90();
    SaveImage(argv[3],Img2);
    
  } else if (strcmp(argv[1],"rotate180")==0) {    
    CheckUsage(argc,4,"rotate180","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    rotate180();
    SaveImage(argv[3],Img2);
    
  } else if (strcmp(argv[1],"rotate270")==0) {    
    CheckUsage(argc,4,"rotate270","in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[2]);
    rotate270();
    SaveImage(argv[3],Img2);

  } else if (strcmp(argv[1],"edge")==0) {    
    CheckUsage(argc,7,"edge","size emptycolor edgecolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[5]);
    edge(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]));
    SaveImage(argv[6],Img0);

  } else if (strcmp(argv[1],"explode")==0) {
    CheckUsage(argc,5,"explode","nbpoints in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    explode(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
             
  } else if (strcmp(argv[1],"palette2rgb")==0) {
    CheckUsage(argc,7,"palette2rgb","in.pal in.g red.g green.g blue.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    palette2rgb(argv[2]);
    SaveImage(argv[4],Img0);
    SaveImage(argv[5],Img1);
    SaveImage(argv[6],Img2);

  } else if (strcmp(argv[1],"textpalette2rgb")==0) {
    CheckUsage(argc,7,"textpalette2rgb","in.txt in.g red.g green.g blue.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    textpalette2rgb(argv[2]);
    SaveImage(argv[4],Img0);
    SaveImage(argv[5],Img1);
    SaveImage(argv[6],Img2);
    
  } else if (strcmp(argv[1],"setmincolor")==0) {
    CheckUsage(argc,5,"setmincolor","newmincolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    setmincolor(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
        
  } else if (strcmp(argv[1],"setmaxcolor")==0) {
    CheckUsage(argc,5,"setmaxcolor","newmaxcolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    setmaxcolor(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"allcolorsabove")==0) {
    CheckUsage(argc,5,"allcolorsabove","mincolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    allcolorsabove(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
        
  } else if (strcmp(argv[1],"allcolorsbelow")==0) {
    CheckUsage(argc,5,"allcolorsbelow","maxcolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    allcolorsbelow(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"allcolorsbetween")==0) {
    CheckUsage(argc,6,"allcolorsbetween","mincolor maxcolor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[4]);
    allcolorsbetween(ReadNumber(argv[2]),ReadNumber(argv[3]));
    SaveImage(argv[5],Img0);
    
  } else if (strcmp(argv[1],"mknoise")==0) {    
    CheckUsage(argc,7,"mknoise","min max width height out.g");
    mknoise(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]));
    SaveImage(argv[6],Img0);
    
  } else if (strcmp(argv[1],"disc")==0) {
    CheckUsage(argc,8,"disc","xcenter ycenter radius color in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[6]);
    disc(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]));
    SaveImage(argv[7],Img0);
    
  } else if (strcmp(argv[1],"rectangle")==0) {
    CheckUsage(argc,9,"rectangle","xleft ybottom xright ytop color in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[7]);
    rectangle(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]));
    SaveImage(argv[8],Img0);
    
  } else if (strcmp(argv[1],"rectangle2")==0) {
    CheckUsage(argc,9,"rectangle2","xleft ybottom width height color in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[7]);
    rectangle2(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]));
    SaveImage(argv[8],Img0);
    
  } else if (strcmp(argv[1],"landscape")==0) {
    CheckUsage(argc,5,"landscape","nbiterations in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    landscape(ReadNumber(argv[2]));
    SaveImage(argv[4],Img1);

  } else if (strcmp(argv[1],"range")==0) {
    CheckUsage(argc,6,"range","colmin colmax in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[4]);
    range(ReadNumber(argv[2]),ReadNumber(argv[3]));
    SaveImage(argv[5],Img0);


  } else if (strcmp(argv[1],"fillwithOs")==0) {
    CheckUsage(argc,11,"fillwithOs","OutRMin OutRMax InRMmin InRMmax Border ClearColor CircleColor in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[9]);
    fillwithOs(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]),ReadNumber(argv[7]),ReadNumber(argv[8]));
    SaveImage(argv[10],Img0);
        
  } else if (strcmp(argv[1],"floydsteinberg")==0) {
    CheckUsage(argc,5,"floydsteinberg","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    floydsteinberg(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"stucki")==0) {
    CheckUsage(argc,5,"stucki","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    stucki(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"ordered1x2")==0) {
    CheckUsage(argc,5,"ordered1x2","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered1x2(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"ordered2x1")==0) {
    CheckUsage(argc,5,"ordered2x1","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered2x1(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);

  } else if (strcmp(argv[1],"ordered2x2")==0) {
    CheckUsage(argc,5,"ordered2x2","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered2x2(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);

  } else if (strcmp(argv[1],"ordered3x3")==0) {
    CheckUsage(argc,5,"ordered3x3","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered3x3(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"ordered4x4")==0) {
    CheckUsage(argc,5,"ordered4x4","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered4x4(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);
    
  } else if (strcmp(argv[1],"ordered4x4bis")==0) {
    CheckUsage(argc,5,"ordered4x4bis","nb_colors in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    ordered4x4bis(ReadNumber(argv[2]));
    SaveImage(argv[4],Img0);

  } else if (strcmp(argv[1],"sampleup_miller")==0) {
    CheckUsage(argc,5,"sampleup_miller","nbiterations in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    sampleup_miller(ReadNumber(argv[2]));
    SaveImage(argv[4],Img1);
    
  } else if (strcmp(argv[1],"sampledown")==0) {
    CheckUsage(argc,6,"sampledown","xratio yratio in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[4]);
    sampledown(ReadNumber(argv[2]),ReadNumber(argv[3]));
    SaveImage(argv[5],Img1);

  } else if (strcmp(argv[1],"scaleup")==0) {
    CheckUsage(argc,6,"scaleup","xratio yratio in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[4]);
    scaleup(ReadNumber(argv[2]),ReadNumber(argv[3]));
    SaveImage(argv[5],Img1);
    
  } else if (strcmp(argv[1],"scaledown")==0) {
    CheckUsage(argc,6,"scaledown","xratio yratio in.g out.g");
    Img0=AllocMemAndLoadGreyImage(argv[4]);
    scaledown(ReadNumber(argv[2]),ReadNumber(argv[3]));
    SaveImage(argv[5],Img1);
    
  } else if (strcmp(argv[1],"mkstrippedtube")==0) {
    CheckUsage(argc,6,"mkstrippedtube","diameter nbsteps nbstripes col0 col1 out.g");
    mkstrippedtube(ReadNumber(argv[2]),ReadNumber(argv[3]),ReadNumber(argv[4]),ReadNumber(argv[5]),ReadNumber(argv[6]));
    SaveImage(argv[7],Img0);

  } else if (strcmp(argv[1],"grey2bitplanes")==0) {
    CheckUsage(argc,3,"grey2bitplanes","nbbitplanes in.g out.xxx");
    Img0=AllocMemAndLoadGreyImage(argv[3]);
    grey2bitplanes(ReadNumber(argv[2]),argv[4]);

  } else {
    printf("Error: %s is not a valid function\nSee AllFunctions.txt\n",argv[1]); exit(1);
  }
  
  printf(" OK");
  exit(0);
}
