; TIATracker music player
; Copyright 2016 Andre "Kylearan" Wichmann
; Website: https://bitbucket.org/kylearan/tiatracker
; Email: andre.wichmann@gmx.de
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
;   http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

; Song author: shiru8bit
; Song name: New Year 2024 intro

; @com.wudsn.ide.asm.hardware=ATARI2600

; =====================================================================
; TIATracker melodic and percussion instruments, patterns and sequencer
; data.
; =====================================================================
tt_TrackDataStart:

; =====================================================================
; Melodic instrument definitions (up to 7). tt_envelope_index_c0/1 hold
; the index values into these tables for the current instruments played
; in channel 0 and 1.
; 
; Each instrument is defined by:
; - tt_InsCtrlTable: the AUDC value
; - tt_InsADIndexes: the index of the start of the ADSR envelope as
;       defined in tt_InsFreqVolTable
; - tt_InsSustainIndexes: the index of the start of the Sustain phase
;       of the envelope
; - tt_InsReleaseIndexes: the index of the start of the Release phase
; - tt_InsFreqVolTable: The AUDF frequency and AUDV volume values of
;       the envelope
; =====================================================================

; Instrument master CTRL values
tt_InsCtrlTable:
        dc.b $06, $04, $0c, $04, $0c


; Instrument Attack/Decay start indexes into ADSR tables.
tt_InsADIndexes:
        dc.b $00, $07, $07, $11, $11


; Instrument Sustain start indexes into ADSR tables
tt_InsSustainIndexes:
        dc.b $03, $0d, $0d, $14, $14


; Instrument Release start indexes into ADSR tables
; Caution: Values are stored with an implicit -1 modifier! To get the
; real index, add 1.
tt_InsReleaseIndexes:
        dc.b $04, $0e, $0e, $15, $15


; AUDVx and AUDFx ADSR envelope values.
; Each byte encodes the frequency and volume:
; - Bits 7..4: Freqency modifier for the current note ([-8..7]),
;       8 means no change. Bit 7 is the sign bit.
; - Bits 3..0: Volume
; Between sustain and release is one byte that is not used and
; can be any value.
; The end of the release phase is encoded by a 0.
tt_InsFreqVolTable:
; 0: bass
        dc.b $8a, $8a, $87, $86, $00, $80, $00
; 1+2: mel
        dc.b $88, $88, $84, $83, $83, $82, $82, $00
        dc.b $80, $00
; 3+4: mel_s
        dc.b $82, $82, $81, $81, $00, $80, $00



; =====================================================================
; Percussion instrument definitions (up to 15)
;
; Each percussion instrument is defined by:
; - tt_PercIndexes: The index of the first percussion frame as defined
;       in tt_PercFreqTable and tt_PercCtrlVolTable
; - tt_PercFreqTable: The AUDF frequency value
; - tt_PercCtrlVolTable: The AUDV volume and AUDC values
; =====================================================================

; Indexes into percussion definitions signifying the first frame for
; each percussion in tt_PercFreqTable.
; Caution: Values are stored with an implicit +1 modifier! To get the
; real index, subtract 1.
tt_PercIndexes:
        dc.b $01, $05, $0b


; The AUDF frequency values for the percussion instruments.
; If the second to last value is negative (>=128), it means it's an
; "overlay" percussion, i.e. the player fetches the next instrument note
; immediately and starts it in the sustain phase next frame. (Needs
; TT_USE_OVERLAY)
tt_PercFreqTable:
; 0: kick
        dc.b $0f, $16, $1a, $00
; 1: hat
        dc.b $00, $00, $00, $00, $00, $00
; 2: snare
        dc.b $16, $03, $03, $18, $19, $1e, $00, $00
        dc.b $00, $00, $00, $00


; The AUDCx and AUDVx volume values for the percussion instruments.
; - Bits 7..4: AUDC value
; - Bits 3..0: AUDV value
; 0 means end of percussion data.
tt_PercCtrlVolTable:
; 0: kick
        dc.b $fd, $fb, $f9, $00
; 1: hat
        dc.b $8a, $83, $80, $80, $80, $00
; 2: snare
        dc.b $c8, $88, $88, $c6, $c4, $c2, $82, $82
        dc.b $81, $81, $80, $00


        
; =====================================================================
; Track definition
; The track is defined by:
; - tt_PatternX (X=0, 1, ...): Pattern definitions
; - tt_PatternPtrLo/Hi: Pointers to the tt_PatternX tables, serving
;       as index values
; - tt_SequenceTable: The order in which the patterns should be played,
;       i.e. indexes into tt_PatternPtrLo/Hi. Contains the sequences
;       for all channels and sub-tracks. The variables
;       tt_cur_pat_index_c0/1 hold an index into tt_SequenceTable for
;       each channel.
;
; So tt_SequenceTable holds indexes into tt_PatternPtrLo/Hi, which
; in turn point to pattern definitions (tt_PatternX) in which the notes
; to play are specified.
; =====================================================================

; ---------------------------------------------------------------------
; Pattern definitions, one table per pattern. tt_cur_note_index_c0/1
; hold the index values into these tables for the current pattern
; played in channel 0 and 1.
;
; A pattern is a sequence of notes (one byte per note) ending with a 0.
; A note can be either:
; - Pause: Put melodic instrument into release. Must only follow a
;       melodic instrument.
; - Hold: Continue to play last note (or silence). Default "empty" note.
; - Slide (needs TT_USE_SLIDE): Adjust frequency of last melodic note
;       by -7..+7 and keep playing it
; - Play new note with melodic instrument
; - Play new note with percussion instrument
; - End of pattern
;
; A note is defined by:
; - Bits 7..5: 1-7 means play melodic instrument 1-7 with a new note
;       and frequency in bits 4..0. If bits 7..5 are 0, bits 4..0 are
;       defined as:
;       - 0: End of pattern
;       - [1..15]: Slide -7..+7 (needs TT_USE_SLIDE)
;       - 8: Hold
;       - 16: Pause
;       - [17..31]: Play percussion instrument 1..15
;
; The tracker must ensure that a pause only follows a melodic
; instrument or a hold/slide.
; ---------------------------------------------------------------------
TT_FREQ_MASK    = %00011111
TT_INS_HOLD     = 8
TT_INS_PAUSE    = 16
TT_FIRST_PERC   = 17

; intro11l
tt_pattern0:
        dc.b $11, $33, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $33, $08, $08, $08
        dc.b $11, $33, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $12, $08, $11, $08
        dc.b $00

; intro12l
tt_pattern1:
        dc.b $11, $3a, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $3a, $08, $08, $08
        dc.b $11, $3a, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $12, $08, $11, $08
        dc.b $00

; introl13l
tt_pattern2:
        dc.b $11, $31, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $31, $08, $08, $08
        dc.b $11, $31, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $31, $08, $2e, $08
        dc.b $00

; intro14l
tt_pattern3:
        dc.b $11, $3d, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $3d, $08, $08, $08
        dc.b $11, $3d, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $11, $08, $13, $08
        dc.b $00

; intro15l
tt_pattern4:
        dc.b $11, $3d, $08, $08, $12, $08, $08, $08
        dc.b $11, $08, $08, $08, $3d, $08, $08, $08
        dc.b $11, $3d, $08, $08, $12, $08, $13, $08
        dc.b $11, $08, $08, $08, $13, $08, $13, $08
        dc.b $00

; mel11l
tt_pattern5:
        dc.b $11, $2e, $08, $08, $12, $08, $08, $08
        dc.b $13, $08, $2e, $10, $2e, $08, $12, $08
        dc.b $11, $2e, $08, $08, $12, $08, $08, $08
        dc.b $13, $08, $2e, $10, $2e, $08, $12, $08
        dc.b $00

; mel13l
tt_pattern6:
        dc.b $11, $33, $08, $08, $12, $08, $08, $08
        dc.b $13, $08, $33, $10, $33, $08, $12, $08
        dc.b $11, $33, $08, $08, $12, $08, $08, $08
        dc.b $13, $08, $33, $10, $33, $08, $12, $08
        dc.b $00

; mel15l
tt_pattern7:
        dc.b $11, $33, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $33, $10, $33, $08, $12, $08
        dc.b $11, $33, $08, $08, $13, $08, $13, $08
        dc.b $13, $08, $13, $08, $33, $08, $08, $08
        dc.b $00

; middle11l
tt_pattern8:
        dc.b $11, $33, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $33, $08, $12, $08
        dc.b $11, $33, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $12, $08, $11, $08
        dc.b $00

; middle12l
tt_pattern9:
        dc.b $11, $3a, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $3a, $08, $12, $08
        dc.b $11, $3a, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $12, $08, $13, $08
        dc.b $00

; middle13l
tt_pattern10:
        dc.b $11, $31, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $31, $08, $12, $08
        dc.b $11, $31, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $31, $08, $2e, $08
        dc.b $00

; middle14l
tt_pattern11:
        dc.b $11, $3d, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $3d, $08, $12, $08
        dc.b $11, $3d, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $13, $08, $13, $08
        dc.b $00

; middle15l
tt_pattern12:
        dc.b $11, $3d, $08, $08, $12, $08, $12, $08
        dc.b $13, $08, $12, $08, $3d, $08, $12, $08
        dc.b $11, $3d, $08, $08, $13, $08, $13, $08
        dc.b $13, $08, $13, $08, $13, $08, $13, $08
        dc.b $00

; mel21l
tt_pattern13:
        dc.b $3d, $08, $10, $08, $3d, $08, $10, $08
        dc.b $3d, $08, $10, $08, $3d, $08, $10, $08
        dc.b $13, $3d, $08, $10, $3d, $08, $10, $08
        dc.b $3d, $08, $10, $08, $3d, $08, $10, $08
        dc.b $00

; mel22l
tt_pattern14:
        dc.b $3a, $08, $10, $08, $3a, $08, $10, $08
        dc.b $3a, $08, $10, $08, $3a, $08, $10, $08
        dc.b $13, $3a, $08, $10, $3a, $08, $10, $08
        dc.b $3a, $08, $10, $08, $3a, $08, $10, $08
        dc.b $00

; mel23l
tt_pattern15:
        dc.b $37, $08, $10, $08, $37, $08, $10, $08
        dc.b $37, $08, $10, $08, $37, $08, $10, $08
        dc.b $13, $37, $08, $10, $37, $08, $10, $08
        dc.b $37, $08, $10, $08, $37, $08, $10, $08
        dc.b $00

; mel24l
tt_pattern16:
        dc.b $31, $08, $10, $08, $31, $08, $10, $08
        dc.b $31, $08, $10, $08, $31, $08, $10, $08
        dc.b $13, $31, $08, $10, $31, $08, $10, $08
        dc.b $31, $08, $10, $08, $13, $08, $13, $08
        dc.b $00

; mel25l
tt_pattern17:
        dc.b $11, $08, $3d, $10, $2e, $08, $3d, $10
        dc.b $13, $08, $3d, $10, $2e, $08, $3d, $10
        dc.b $11, $08, $3d, $10, $11, $08, $3d, $10
        dc.b $13, $08, $3d, $10, $2e, $08, $3d, $10
        dc.b $00

; mel26l
tt_pattern18:
        dc.b $11, $08, $3a, $10, $3a, $08, $3a, $10
        dc.b $13, $08, $3a, $10, $3a, $08, $3a, $10
        dc.b $11, $08, $3a, $10, $11, $08, $3a, $10
        dc.b $13, $08, $3a, $10, $3a, $08, $3a, $10
        dc.b $00

; mel27l
tt_pattern19:
        dc.b $11, $08, $33, $10, $29, $08, $33, $10
        dc.b $13, $08, $33, $10, $29, $08, $33, $10
        dc.b $11, $08, $33, $10, $11, $08, $33, $10
        dc.b $13, $08, $33, $10, $29, $08, $33, $10
        dc.b $00

; mel28l
tt_pattern20:
        dc.b $11, $08, $33, $10, $29, $08, $33, $10
        dc.b $13, $08, $33, $10, $29, $08, $33, $10
        dc.b $13, $08, $08, $08, $13, $74, $08, $08
        dc.b $70, $08, $ae, $08, $6c, $08, $13, $08
        dc.b $00

; intro11r
tt_pattern21:
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $5e, $08, $99, $08, $5c, $08, $9e, $08
        dc.b $00

; intro12r
tt_pattern22:
        dc.b $59, $08, $9c, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $54, $08, $99, $08, $52, $08, $94, $08
        dc.b $00

; intro13r
tt_pattern23:
        dc.b $56, $08, $92, $08, $56, $08, $96, $08
        dc.b $56, $08, $96, $08, $56, $08, $96, $08
        dc.b $56, $08, $96, $08, $56, $08, $96, $08
        dc.b $54, $08, $96, $08, $52, $08, $94, $08
        dc.b $00

; intro14r
tt_pattern24:
        dc.b $59, $08, $92, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $72, $08, $99, $08
        dc.b $6e, $08, $b2, $08, $6c, $08, $ae, $08
        dc.b $00

; mel11r
tt_pattern25:
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $56, $08, $99, $08, $59, $08, $96, $08
        dc.b $96, $08, $99, $08, $6c, $08, $99, $08
        dc.b $5e, $08, $ac, $08, $9e, $08, $9e, $08
        dc.b $00

; mel12r
tt_pattern26:
        dc.b $59, $08, $9e, $08, $59, $08, $99, $08
        dc.b $56, $08, $99, $08, $59, $08, $96, $08
        dc.b $99, $08, $99, $08, $6c, $08, $99, $08
        dc.b $5e, $08, $ac, $08, $5c, $08, $9e, $08
        dc.b $00

; mel13r
tt_pattern27:
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $56, $08, $99, $08, $54, $08, $96, $08
        dc.b $00

; mel14r
tt_pattern28:
        dc.b $59, $08, $94, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $59, $08, $99, $08, $59, $08, $99, $08
        dc.b $5c, $08, $99, $08, $5e, $08, $9c, $08
        dc.b $00

; mel21r
tt_pattern29:
        dc.b $10, $08, $08, $08, $5e, $08, $9e, $08
        dc.b $59, $08, $9e, $08, $5e, $08, $99, $08
        dc.b $6e, $08, $9e, $08, $6c, $08, $ae, $08
        dc.b $ac, $08, $08, $08, $72, $08, $ac, $08
        dc.b $00

; mel22r
tt_pattern30:
        dc.b $10, $08, $08, $08, $5e, $08, $9e, $08
        dc.b $59, $08, $9e, $08, $5e, $08, $99, $08
        dc.b $6e, $08, $9e, $08, $6c, $08, $ae, $08
        dc.b $ac, $08, $08, $08, $52, $56, $52, $56
        dc.b $00

; mel23r
tt_pattern31:
        dc.b $92, $96, $92, $96, $56, $59, $56, $59
        dc.b $96, $99, $96, $99, $10, $08, $08, $08
        dc.b $5e, $08, $08, $08, $9e, $08, $08, $08
        dc.b $59, $08, $96, $08, $6c, $08, $99, $08
        dc.b $00




; Individual pattern speeds (needs TT_GLOBAL_SPEED = 0).
; Each byte encodes the speed of one pattern in the order
; of the tt_PatternPtr tables below.
; If TT_USE_FUNKTEMPO is 1, then the low nibble encodes
; the even speed and the high nibble the odd speed.
    IF TT_GLOBAL_SPEED = 0
tt_PatternSpeeds:
%%PATTERNSPEEDS%%
    ENDIF


; ---------------------------------------------------------------------
; Pattern pointers look-up table.
; ---------------------------------------------------------------------
tt_PatternPtrLo:
        dc.b <tt_pattern0, <tt_pattern1, <tt_pattern2, <tt_pattern3
        dc.b <tt_pattern4, <tt_pattern5, <tt_pattern6, <tt_pattern7
        dc.b <tt_pattern8, <tt_pattern9, <tt_pattern10, <tt_pattern11
        dc.b <tt_pattern12, <tt_pattern13, <tt_pattern14, <tt_pattern15
        dc.b <tt_pattern16, <tt_pattern17, <tt_pattern18, <tt_pattern19
        dc.b <tt_pattern20, <tt_pattern21, <tt_pattern22, <tt_pattern23
        dc.b <tt_pattern24, <tt_pattern25, <tt_pattern26, <tt_pattern27
        dc.b <tt_pattern28, <tt_pattern29, <tt_pattern30, <tt_pattern31

tt_PatternPtrHi:
        dc.b >tt_pattern0, >tt_pattern1, >tt_pattern2, >tt_pattern3
        dc.b >tt_pattern4, >tt_pattern5, >tt_pattern6, >tt_pattern7
        dc.b >tt_pattern8, >tt_pattern9, >tt_pattern10, >tt_pattern11
        dc.b >tt_pattern12, >tt_pattern13, >tt_pattern14, >tt_pattern15
        dc.b >tt_pattern16, >tt_pattern17, >tt_pattern18, >tt_pattern19
        dc.b >tt_pattern20, >tt_pattern21, >tt_pattern22, >tt_pattern23
        dc.b >tt_pattern24, >tt_pattern25, >tt_pattern26, >tt_pattern27
        dc.b >tt_pattern28, >tt_pattern29, >tt_pattern30, >tt_pattern31
        


; ---------------------------------------------------------------------
; Pattern sequence table. Each byte is an index into the
; tt_PatternPtrLo/Hi tables where the pointers to the pattern
; definitions can be found. When a pattern has been played completely,
; the next byte from this table is used to get the address of the next
; pattern to play. tt_cur_pat_index_c0/1 hold the current index values
; into this table for channels 0 and 1.
; If TT_USE_GOTO is used, a value >=128 denotes a goto to the pattern
; number encoded in bits 6..0 (i.e. value AND %01111111).
; ---------------------------------------------------------------------
tt_SequenceTable:
        ; ---------- Channel 0 ----------
        dc.b $00, $01, $02, $03, $00, $01, $02, $04
        dc.b $05, $05, $06, $06, $05, $05, $06, $07
        dc.b $08, $09, $0a, $0b, $08, $09, $0a, $0c
        dc.b $0d, $0e, $0f, $10, $11, $12, $13, $14
        dc.b $80

        
        ; ---------- Channel 1 ----------
        dc.b $15, $16, $17, $18, $15, $16, $17, $18
        dc.b $19, $1a, $1b, $1c, $19, $1a, $1b, $1c
        dc.b $15, $16, $17, $18, $15, $16, $17, $18
        dc.b $1d, $1d, $1e, $1f, $1d, $1d, $1e, $1f
        dc.b $a1


        echo "Track size: ", *-tt_TrackDataStart
